/* 
 *  InteractionExecutor.java
 * 
 *  Copyright 2016 Avaya Inc. All Rights Reserved.
 * 
 *  Usage of this source is bound to the terms described
 *  in AvayaLicenseSDK.rtf.
 * 
 *  Avaya - Confidential & Proprietary. Use pursuant to your signed agreement
 *  or Avaya Policy
 * 
 */
package com.avaya.ccs.javafxrefclient;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.avaya.ccs.api.ContactCenterCodeI;
import com.avaya.ccs.api.ContactNumberI;
import com.avaya.ccs.api.DestinationI;
import com.avaya.ccs.api.EmailListEntryI;
import com.avaya.ccs.api.InteractionI;
import com.avaya.ccs.api.InteractionListenerI;
import com.avaya.ccs.api.MediaI;
import com.avaya.ccs.api.EmailMediaI;
import com.avaya.ccs.api.NotificationEventI;
import com.avaya.ccs.api.PhoneListEntryI;
import com.avaya.ccs.api.PostalAddressListEntryI;
import com.avaya.ccs.api.enums.ContactType;
import com.avaya.ccs.api.enums.InteractionState;
import com.avaya.ccs.api.enums.NotificationType;
import com.avaya.ccs.api.exceptions.ObjectInvalidException;
import com.avaya.ccs.core.CommandArgs;
import com.avaya.ccs.core.Interaction;
import com.avaya.ccs.javafxrefclient.request.*;

public class InteractionExecutor extends Executor implements InteractionListenerI {

    private static final Logger LOG = Logger.getLogger(InteractionExecutor.class);
    private final JavaFXRefClient gui;
    private static final InteractionTimerMap TIMERS = new InteractionTimerMap();

    public InteractionExecutor(JavaFXRefClient gui) {
        this.gui = gui;
    }

    InteractionMediaEventTask interactionMediaEventTask;
    InteractionEventTask interactionEventTask;
    
    @Override
    public void onInteractionEvent(NotificationEventI<InteractionI> nei) {
        String methodName = "onInteractionEvent() ";
        LOG.info(methodName + nei.getNotificationType() + " InteractionI-ID:" + nei.getNotificationObject().getId());
        Interaction i = (Interaction) nei.getNotificationObject();
       
    	if (nei.getNotificationType() == NotificationType.NEW || nei.getNotificationType() == NotificationType.UPDATE) {
    		try {
        		if (i.canOpenMedia()) {
        			i.openMedia();
        		}
	        } catch (ObjectInvalidException e) {
				LOG.error(methodName, e);
			}
    	}
        
        executeIncomingEvent(new InteractionEventTask(nei, gui.getInteractionDataGUI(), gui.getSupervisorGUI(), this));    
    }
    
    @Override
    public void onInteractionMediaEvent(NotificationEventI<? extends MediaI> nei) {
    	String methodName = "onInteractionMediaEvent()";
    	LOG.info(methodName + nei.getNotificationType() + " MediaI-ID: " + nei.getNotificationObject().getId() + ". Contact Type:" + nei.getNotificationObject().getContactType());
//    	ContactType contactType=  ContactType.get(nei.getNotificationObject().getContactType());
    	interactionMediaEventTask = new InteractionMediaEventTask(nei, gui.getInteractionDataGUI(), gui.getSupervisorGUI(), nei.getNotificationObject().getContactType());
		executeIncomingEvent(interactionMediaEventTask);
		
    }

    public void answer(String interactionID) {
        String methodName = "answer() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new AnswerRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void end(String interactionID) {
        String methodName = "end() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new EndRequest(interactionID));
        LOG.trace(methodName + "-");
    }
    
    public void end(String interactionId, String closeReasonCode, String closingAgentNote, boolean sendTranscript) {
        String methodName = "end() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionId);
        executeOutgoingRequest(new EndRequest(interactionId, closeReasonCode, closingAgentNote, sendTranscript));
        LOG.trace(methodName + "-");
    }

    public void hold(String interactionID) {
        String methodName = "hold() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new HoldRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void unHold(String interactionID) {
        String methodName = "unHold() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new UnHoldRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void initTransfer(String interactionID, String destination) {
        String methodName = "initTransfer() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Destination:" + destination);
        executeOutgoingRequest(new InitTransferRequest(interactionID, destination));
        LOG.trace(methodName + "-");
    }

    public void completeTransfer(String interactionID) {
        String methodName = "completeTransfer() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new CompleteTransferRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void initConference(String interactionID, String destination) {
        String methodName = "initConference() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Destination:" + destination);
        executeOutgoingRequest(new InitConferenceRequest(interactionID, destination));
        LOG.trace(methodName + "-");
    }

    public void completeConference(String interactionID) {
        String methodName = "completeConference() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new CompleteConferenceRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void join(String sourceInteractionID, String destinationInteractionID) {
        String methodName = "join() ";
        LOG.debug(methodName + "+ SourceInteractionID:" + sourceInteractionID + " DestinationInteractionID:"
                + destinationInteractionID);
        executeOutgoingRequest(new JoinRequest(sourceInteractionID, destinationInteractionID));
        LOG.trace(methodName + "-");
    }

    public void getInteractionData() {
        String methodName = "getInteractionData() ";
        LOG.debug(methodName + "+");
        executeOutgoingRequest(new GetInteractionDataRequest(gui.getInteractionDataGUI()));
        LOG.trace(methodName + "-");
    }
    
    public void openInteractionMedia(String interactionId) {
    	String methodName="openInteractionMedia() ";
        LOG.info(methodName + "+");
    	executeOutgoingRequest(new OpenMediaDataRequest(interactionId));
    	LOG.info(methodName + "-");
    }
   
    public void setActivityCode(String interactionID, String code) {
        String methodName = "setActivityCode() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Code:" + code);
        executeOutgoingRequest(new SetActivityCodeRequest(interactionID, code));
        LOG.trace(methodName + "-");
    }

    public void getActivityCodes(String interactionID) {
        String methodName = "getActivityCodes() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new GetActivityCodesRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void getCustomFields(String interactionID) {
        String methodName = "getCustomFields() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new GetCustomFieldsRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void getDispositionCodes(String interactionID) {
        String methodName = "getDispostionCodes() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new GetDispostionCodesRequest(interactionID));
        LOG.trace(methodName + "-");
    }
    
    public void getAutoPhrases(String interactionID) {
        String methodName = "getAutoPhrases() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new GetAutoPhrasesRequest(interactionID));
        LOG.trace(methodName + "-");
    }
    
    public void getPagePushUrls(String interactionID) {
        String methodName = "getPagePushUrls() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new GetPagePushUrlsRequest(interactionID));
        LOG.trace(methodName + "-");
    }
    

    public void getCustomerHistory(String interactionID) {
        String methodName = "requestCustomerHistory() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new GetCustomerHistoryRequest(interactionID));
        LOG.trace(methodName + "-");
    }
    public void getApprovalStatus(String interactionID) {
        String methodName = "getApprovalStatus() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new GetApprovalStatusRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void reviewEmail(String interactionID, String reviewType, Map<CommandArgs.Argument,
            String> queryParams, String data) {
        String methodName = "reviewEmail() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new ReviewEmailRequest(interactionID, reviewType, queryParams, data));
        LOG.trace(methodName + "-");
    }

    public void editStartEmail(String interactionID, String actionId) {
        String methodName = "editStartEmail() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new ReviewEmailEditStartRequest(interactionID, actionId));
        LOG.trace(methodName + "-");
    }
    public void reviewApprovedEmail(String interactionID, Long approvalId, String comment) {
        String methodName = "reviewApprovedEmail() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new ReviewEmailReviewApprovedRequest(interactionID, approvalId.toString(), comment));
        LOG.trace(methodName + "-");
    }

    public void reviewRejectedEmail(String interactionID, Long approvalId, String comment) {
        String methodName = "reviewRejectedEmail() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new ReviewEmailReviewRejectRequest(interactionID, approvalId.toString(), comment));
        LOG.trace(methodName + "-");
    }

    
    public void getCustomerInteractionMedia(String interactionId, String queryContactID) {
        String methodName = "requestCustomerInteractionMedia() ";
        LOG.debug(methodName + "+ contactID:" + queryContactID);
        executeOutgoingRequest(new GetCustomerInteractionMediaRequest(interactionId,queryContactID));
        LOG.trace(methodName + "-");
    }

    public void removeIntrinsic(String interactionID, String intrinsicName) {
        String methodName = "removeIntrinsic() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " IntrinsicName:" + intrinsicName);
        executeOutgoingRequest(new RemoveIntrinsicRequest(interactionID, intrinsicName));
        LOG.trace(methodName + "-");
    }
   
    public void editOrAddIntrinsic(String interactionID, String intrinsicName, String newValue) {
        String methodName = "editOrAddIntrinsic() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " IntrinsicName:" + intrinsicName + " NewValue:"
                + newValue);
        executeOutgoingRequest(new EditOrAddIntrinsicRequest(interactionID, intrinsicName, newValue));
        LOG.trace(methodName + "-");
    }

    public void setUUI(String interactionID, String newValue) {
        String methodName = "setUUI() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " NewValue:" + newValue);
        executeOutgoingRequest(new SetUUIRequest(interactionID, newValue));
        LOG.trace(methodName + "-");
    }

    public void setDATA(String interactionID, String newValue) {
        String methodName = "setDATA() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " NewValue:" + newValue);
        executeOutgoingRequest(new SetDataRequest(interactionID, newValue));
        LOG.trace(methodName + "-");
    }

    public void setDATA(String interactionID, Map<String, String> newValue) {
        String methodName = "setDATA() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new SetDataRequest(interactionID, newValue));
        LOG.trace(methodName + "-");
    }

    public void setDATA(String interactionID, byte[] newValue) {
        String methodName = "setDATA() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " ByteArraySize:" + newValue.length);
        executeOutgoingRequest(new SetDataRequest(interactionID, newValue));
        LOG.trace(methodName + "-");
    }

    public void addKVPDATA(String interactionID, String newName, String newValue) {
        String methodName = "addKVPDATA() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " NewName:" + newName + " NewValue:" + newValue);
        executeOutgoingRequest(new AddKVPToExistingDataRequest(interactionID, newName, newValue));
        LOG.trace(methodName + "-");
    }

    public void removeKVPDATA(String interactionID, String name) {
        String methodName = "removeKVPDATA() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Name:" + name);
        executeOutgoingRequest(new RemoveKVPDataRequest(interactionID, name));
        LOG.trace(methodName + "-");
    }

    public void editKVPDATA(String interactionID, String name, String newValue) {
        String methodName = "editKVPDATA() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Name:" + name + " NewValue:" + newValue);
        executeOutgoingRequest(new EditKVPValueRequest(interactionID, name, newValue));
        LOG.trace(methodName + "-");
    }

    public void deleteDATA(String interactionID) {
        String methodName = "deleteDATA() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new RemoveDataRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void observe(String superID, String userID, String interactionID) {
        String methodName = "observe() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " SupervisorID:" + superID + " UserID:" + userID);
        executeOutgoingRequest(new ObserveRequest(superID, userID, interactionID));
        LOG.trace(methodName + "-");
    }

    public void bargeIn(String superID, String userID, String interactionID) {
        String methodName = "bargeIn() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " SupervisorID:" + superID + " UserID:" + userID);
        executeOutgoingRequest(new BargeInRequest(superID, userID, interactionID));
        LOG.trace(methodName + "-");
    }

    public void whisper(String superID, String userID, String interactionID) {
        String methodName = "whisper() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " SupervisorID:" + superID + " UserID:" + userID);
        executeOutgoingRequest(new WhisperRequest(superID, userID, interactionID));
        LOG.trace(methodName + "-");
    }

    public void playDTMF(String digits, String interactionID) {
        String methodName = "playDTMF() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Digits:" + digits);
        executeOutgoingRequest(new PlayDTMFRequest(digits, interactionID));
        LOG.trace(methodName + "-");
    }

    public void emergency(String interactionID) {
        String methodName = "emergency() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new EmergencyRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    // Pom operations
    public void wrapUP(String interactionID) {
        String methodName = "wrapUP() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new WrapUpRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void wrapUPWithCode(String interactionID, ContactCenterCodeI code) {
        String methodName = "wrapUPWithCode() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Code-Name:" + code.getName() + "Number:"
                + code.getNumber());
        executeOutgoingRequest(new WrapUPWithCodeRequest(interactionID, code));
        LOG.trace(methodName + "-");
    }

    public void extendWrapup(String interactionID) {
        String methodName = "extendWrapup() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new ExtendWrapupRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void previewDial(String interactionID, ContactNumberI toDial) {
        String methodName = "previewDial() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " ToDial:" + toDial);
        executeOutgoingRequest(new PreviewDialRequest(interactionID, toDial));
        LOG.trace(methodName + "-");
    }

    public void previewDialFreeForm(String interactionID, String toDial) {
        String methodName = "previewDialFreeForm() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " ToDial:" + toDial);
        executeOutgoingRequest(new PreviewDialFreeFormRequest(interactionID, toDial));
        LOG.trace(methodName + "-");
    }

    public void previewCancel(String interactionID) {
        String methodName = "previewCancel() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new PreviewCancelRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void updateCustomerName(String interactionID, String title, String firstName, String lastName) {
        String methodName = "updateCustomerName() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Title:" + title + " FirstName:" + firstName
                + " LastName:" + lastName);
        executeOutgoingRequest(new UpdateCustomerNameRequest(interactionID, title, firstName, lastName));
        LOG.trace(methodName + "-");
    }

    public void updateCustomerEmail(String interactionID, EmailListEntryI emailEntry) {
        String methodName = "updateCustomerEmail() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " EmailEntry:" + emailEntry);
        executeOutgoingRequest(new UpdateCustomerEmailRequest(interactionID, emailEntry));
        LOG.trace(methodName + "-");
    }

    public void updateCustomerPhone(String interactionID, PhoneListEntryI phoneEntry) {
        String methodName = "updateCustomerPhone() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Phone:" + phoneEntry);
        executeOutgoingRequest(new UpdateCustomerPhoneRequest(interactionID, phoneEntry));
        LOG.trace(methodName + "-");
    }

    public void updateCustomerAddress(String interactionID, PostalAddressListEntryI address) {
        String methodName = "updateCustomerAddress() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " PostalAddress:" + address);
        executeOutgoingRequest(new UpdateCustomerAddressRequest(interactionID, address));
        LOG.trace(methodName + "-");
    }

    public void addToDNC(String interactionID) {
        String methodName = "addToDNC() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new AddToDNCRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    
    public void getConsultDestinationTypes(String interactionID) {
    	String methodName = "getConsultDestinationTypes() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new GetConsultDestinationTypesRequest(interactionID));
        LOG.trace(methodName + "-");
		
	}
    
    public void getConsultDestinations(String interactionID) {
        String methodName = "getConsultDestinations() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new GetConsultDestinationsRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void getCallbackDestinations(String interactionID) {
        String methodName = "getCallbackDestinations() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new GetCallbackDestinationsRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void transferToConsultDestination(String interactionID, DestinationI destination) {
        String methodName = "transferToConsultDestination() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Desination:" + destination);
        executeOutgoingRequest(new TransferToDestinationRequest(interactionID, destination));
        LOG.trace(methodName + "-");
    }
    
    public void initiateTransferToConsultDestination(String interactionID, DestinationI destination) {
        String methodName = "initiateTransferToConsultDestination() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Desination:" + destination);
        executeOutgoingRequest(new InitiateTransferToDestinationRequest(interactionID, destination));
        LOG.trace(methodName + "-");
    }
    

    public void conferenceToConsultDestination(String interactionID, DestinationI destination) {
        String methodName = "conferenceToConsultDestination() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Desination:" + destination);
        executeOutgoingRequest(new ConferenceToDestinationRequest(interactionID, destination));
        LOG.trace(methodName + "-");
    }
    

    public void endConsult(String interactionID) {
    	String methodName = "endConsult() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new EndConsultRequest(interactionID));
        LOG.trace(methodName + "-");
	}

    public void refreshAgentNotes(String interactionID) {
        String methodName = "refreshAgentNotes() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new RefreshAgentNotes(interactionID));
        LOG.trace(methodName + "-");
    }

    public void addAgentNote(String interactionID, String note) {
        String methodName = "addAgentNote() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Note:" + note);
        executeOutgoingRequest(new AddAgentNote(interactionID, note));
        LOG.trace(methodName + "-");
    }

    public void createCallbackToContactNumber(String interactionID, DestinationI destination,
            ContactNumberI contactNumber, String callbackTime, String callbackExpiryTime, String callbackNote) {
        String methodName = "createCallbackToContactNumber() ";
        String toLog = methodName + "+ callbackDestination:" + destination.getNumber() + " callbackTime:" + callbackTime
                + " callbackExpiryTime:" + callbackExpiryTime + " callbackNote:" + callbackNote + "contactNumber:";
        if (contactNumber != null) {
            toLog += contactNumber.toString();
        } else {
            toLog += "NULL";
        }
        LOG.debug(toLog);

        executeOutgoingRequest(new CallbackRequestToContactNumber(interactionID, destination, contactNumber,
                callbackTime, callbackExpiryTime, callbackNote));
        LOG.trace(methodName + "-");
    }
  
    public void createCallbackToExternalNumber(String interactionID, DestinationI destination, String externalNumber,
            String callbackTime, String callbackExpiryTime, String callbackNote) {
        String methodName = "createCallbackToExternalNumber() ";
        LOG.debug(methodName + "+ callbackDestination:" + destination.getNumber() + " externalNumber:" + externalNumber
                + " callbackTime:" + callbackTime + " callbackExpiryTime:" + callbackExpiryTime + " callbackNote:"
                + callbackNote);

        executeOutgoingRequest(new CallbackRequestToExternalNumber(interactionID, destination, externalNumber,
                callbackTime, callbackExpiryTime, callbackNote));
        LOG.trace(methodName + "-");
    }

    public void changeConferenceOwner(String interactionID) {
        String methodName = "changeConferenceOwner() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new ChangeConfOwnerRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void endConference(String interactionID) {
        String methodName = "endConference() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new EndConferenceRequest(interactionID));
        LOG.trace(methodName + "-");
    }

    public void redial(String interactionID, ContactNumberI number) {
        String methodName = "redial() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID + " Number:" + number);
        executeOutgoingRequest(new RedialRequest(interactionID, number));
        LOG.trace(methodName + "-");
    }

    public void rescheduleEmail(String interactionID, String date, String time, String agentNote) {
        String methodName = "rescheduleEmail() ";
        LOG.debug(methodName + "+ InteractionID:" + interactionID);
        executeOutgoingRequest(new EmailCallbackRequest(interactionID, date, time, agentNote));
        LOG.trace(methodName + "-");
    }
    
    public void searchCustomers(String query) {
    	 String methodName = "searchCustomers";
    	 LOG.debug(methodName + "+ Query:" + query);
		executeOutgoingRequest(new CustomerSearchRequest(query));
	}
    
    public void searchContacts(String query) {
    	String methodName = "searchContacts";
    	LOG.debug(methodName + "+ Query:" + query);
		executeOutgoingRequest(new ContactSearchRequest(query));		
	}


    public static InteractionTimerMap getTimers() {
        return TIMERS;
    }


	public void sendWebChatMessage(String interactionId, String message, boolean isWhisper) {
		final String methodName = "sendWebChatMessage";
		LOG.debug(methodName);
		executeOutgoingRequest(new WebChatMessageRequest(interactionId, message, isWhisper));
	}
	
	public void sendWebChatPagePush(String interactionId, String url) {
		final String methodName = "sendWebChatPagePush";
		LOG.debug(methodName);
		executeOutgoingRequest(new PagePushRequest(interactionId, url));
	}
	
	
	public void sendAgentIsTypingMessage(String interactionId) {
		LOG.debug("sendAgentIsTypingMessage ", interactionId);
		executeOutgoingRequest(new IsTypingRequest(interactionId));
	}
	
	public void getSkillsets(String interactionId) {
		LOG.debug("getSkillsets " + interactionId);
		executeOutgoingRequest(new GetSkillsetsRequest(interactionId));
	}

}
